

class ChatBot {
    constructor() {
        this.isOpen = false;
        this.messages = [];
        this.init();
    }

    init() {

        this.createChatUI();
        this.setupEventListeners();
        this.loadHistory();

        setTimeout(() => {
            this.addBotMessage('Здравствуйте! Я помощник МИАЦ Кургана. Чем могу помочь?');
        }, 1000);
    }

    createChatUI() {

        this.chatContainer = document.getElementById('chatbotContainer');
        this.chatMessages = document.getElementById('chatbotMessages');
        this.chatInput = document.getElementById('chatInput');
        this.sendButton = document.getElementById('sendMessage');
        this.toggleButton = document.getElementById('chatBotToggle');
        this.closeButton = document.getElementById('closeChat');

        this.chatContainer.style.display = 'none';
    }

    setupEventListeners() {

        this.toggleButton?.addEventListener('click', () => this.toggleChat());
        this.closeButton?.addEventListener('click', () => this.closeChat());

        this.sendButton?.addEventListener('click', () => this.sendMessage());
        this.chatInput?.addEventListener('keypress', (e) => {
            if (e.key === 'Enter') this.sendMessage();
        });

        document.getElementById('newsChatBtn')?.addEventListener('click', () => {
            this.openChat();
            this.addBotMessage('Привет! Вы запустили меня из новостей. Чем могу помочь?');
        });
        
        document.getElementById('mobileChatBtn')?.addEventListener('click', () => {
            this.openChat();
        });
        
        document.getElementById('chatBotBtn')?.addEventListener('click', () => {
            this.openChat();
        });
    }

    toggleChat() {
        this.isOpen = !this.isOpen;
        this.chatContainer.style.display = this.isOpen ? 'block' : 'none';
        
        if (this.isOpen) {
            this.chatInput.focus();
        }
    }

    openChat() {
        this.isOpen = true;
        this.chatContainer.style.display = 'block';
        this.chatInput.focus();
    }

    closeChat() {
        this.isOpen = false;
        this.chatContainer.style.display = 'none';
    }

    sendMessage() {
        const message = this.chatInput.value.trim();
        if (!message) return;

        this.addUserMessage(message);

        this.chatInput.value = '';

        setTimeout(() => {
            this.processMessage(message);
        }, 500);
    }

    addUserMessage(text) {
        const messageDiv = document.createElement('div');
        messageDiv.className = 'message user-message';
        messageDiv.innerHTML = `
            <div class="message-content">${this.escapeHtml(text)}</div>
            <div class="message-time">${this.getCurrentTime()}</div>
        `;
        
        this.chatMessages.appendChild(messageDiv);
        this.scrollToBottom();
        this.saveMessage('user', text);
    }

    addBotMessage(text, options = []) {
        const messageDiv = document.createElement('div');
        messageDiv.className = 'message bot-message';
        
        let messageHTML = `
            <div class="message-content">${this.escapeHtml(text)}</div>
            <div class="message-time">${this.getCurrentTime()}</div>
        `;

        if (options.length > 0) {
            messageHTML += '<div class="quick-replies">';
            options.forEach(option => {
                messageHTML += `<button class="btn btn-sm btn-outline-primary me-2 mb-2 quick-reply">${option}</button>`;
            });
            messageHTML += '</div>';
        }
        
        messageDiv.innerHTML = messageHTML;
        
        this.chatMessages.appendChild(messageDiv);
        this.scrollToBottom();
        this.saveMessage('bot', text);

        if (options.length > 0) {
            setTimeout(() => {
                messageDiv.querySelectorAll('.quick-reply').forEach(button => {
                    button.addEventListener('click', (e) => {
                        const reply = e.target.textContent;
                        this.addUserMessage(reply);
                        setTimeout(() => this.processMessage(reply), 500);
                    });
                });
            }, 100);
        }
    }

    processMessage(message) {
        const lowerMessage = message.toLowerCase();

        if (this.containsAny(lowerMessage, ['привет', 'здравствуйте', 'добрый', 'hello'])) {
            this.addBotMessage('Здравствуйте! Рад вас видеть. Вы можете спросить меня о: услугах, горячей линии, справках, обучении или аналитике.');
        }
        else if (this.containsAny(lowerMessage, ['справка', 'справку', 'документ'])) {
            this.addBotMessage('Для заказа справок:\n1. Через онлайн-форму на странице "Для населения"\n2. По телефону горячей линии: 8-800-123-45-67\n3. В поликлинике по месту жительства', 
                ['Заказать справку', 'Горячая линия']);
        }
        else if (this.containsAny(lowerMessage, ['горячая', 'линия', 'телефон', 'позвонить'])) {
            this.addBotMessage('Горячая линия МИАЦ: 8-800-123-45-67\nРежим работы: Пн-Пт 8:00-20:00, Сб-Вс 9:00-18:00\nБесплатно по всей России', 
                ['Позвонить сейчас', 'Другие контакты']);
        }
        else if (this.containsAny(lowerMessage, ['курс', 'обучение', 'нмо', 'повышение'])) {
            this.addBotMessage('Мы предлагаем курсы:\n1. Медицинская статистика (144 часа)\n2. Информатика в здравоохранении (256 часов)\n3. Организация здравоохранения (504 часа)\nПодробнее на странице "Обучение"', 
                ['Посмотреть курсы', 'Записаться']);
        }
        else if (this.containsAny(lowerMessage, ['статистика', 'аналитика', 'данные', 'отчёт'])) {
            this.addBotMessage('Аналитические данные доступны на странице "Аналитика". Там вы найдете:\n1. Динамику заболеваемости\n2. Статистику обращений\n3. Годовые отчёты\n4. Интерактивные графики', 
                ['Открыть аналитику', 'Скачать отчёт']);
        }
        else if (this.containsAny(lowerMessage, ['адрес', 'контакты', 'найти', 'карта'])) {
            this.addBotMessage('Наш адрес: г. Курган, ул. Медицинская, 123\nТелефон: +7 (3522) 12-34-56\nEmail: info@miac-kurgan.ru\nСтраница с контактами содержит полную информацию и карту.', 
                ['Открыть контакты', 'Посмотреть на карте']);
        }
        else if (this.containsAny(lowerMessage, ['сайт', 'навигация', 'разделы', 'меню'])) {
            this.addBotMessage('Основные разделы сайта:\n• О центре - миссия, структура\n• Услуги - для организаций и населения\n• Аналитика - статистика и отчёты\n• Обучение - курсы НМО\n• Документы - формы и нормативы\n• Контакты - как связаться');
        }
        else if (this.containsAny(lowerMessage, ['работа', 'график', 'часы', 'открыто'])) {
            this.addBotMessage('Режим работы МИАЦ:\n• Пн-Чт: 8:30-17:30\n• Пт: 8:30-16:15\n• Сб-Вс: выходной\n• Перерыв: 12:30-13:30\nГорячая линия работает ежедневно!');
        }
        else {
            this.addBotMessage('Извините, я не совсем понял ваш вопрос. Попробуйте задать его другими словами или выберите одну из тем:\n1. Горячая линия\n2. Заказ справок\n3. Обучение\n4. Контакты', 
                ['Горячая линия', 'Справки', 'Обучение', 'Контакты']);
        }
    }

    containsAny(text, keywords) {
        return keywords.some(keyword => text.includes(keyword));
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML.replace(/\n/g, '<br>');
    }

    getCurrentTime() {
        const now = new Date();
        return now.toLocaleTimeString('ru-RU', { hour: '2-digit', minute: '2-digit' });
    }

    scrollToBottom() {
        this.chatMessages.scrollTop = this.chatMessages.scrollHeight;
    }

    saveMessage(sender, text) {
        this.messages.push({
            sender: sender,
            text: text,
            time: new Date().toISOString()
        });

        if (this.messages.length > 50) {
            this.messages = this.messages.slice(-50);
        }
        
        localStorage.setItem('chatbot_history', JSON.stringify(this.messages));
    }

    loadHistory() {
        try {
            const saved = localStorage.getItem('chatbot_history');
            if (saved) {
                this.messages = JSON.parse(saved);
            }
        } catch (e) {
            console.error('Ошибка загрузки истории чата:', e);
        }
    }
}

document.addEventListener('DOMContentLoaded', () => {
    window.chatBot = new ChatBot();
});