

document.addEventListener('DOMContentLoaded', function() {
    initComponents();
    setupEventListeners();
});

function initComponents() {

    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });

    var popoverTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="popover"]'));
    var popoverList = popoverTriggerList.map(function (popoverTriggerEl) {
        return new bootstrap.Popover(popoverTriggerEl);
    });

    setupScrollAnimations();
}

function setupEventListeners() {

    const forms = document.querySelectorAll('form');
    forms.forEach(form => {
        form.addEventListener('submit', handleFormSubmit);
    });

    document.querySelectorAll('a[href^="#"]').forEach(anchor => {
        anchor.addEventListener('click', smoothScroll);
    });

    document.querySelectorAll('.show-more-btn').forEach(btn => {
        btn.addEventListener('click', toggleMoreContent);
    });
}

function handleFormSubmit(event) {
    event.preventDefault();
    const form = event.target;
    const submitBtn = form.querySelector('button[type="submit"]');

    const originalText = submitBtn.textContent;
    submitBtn.innerHTML = '<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span> Отправка...';
    submitBtn.disabled = true;

    setTimeout(() => {

        alert('Форма отправлена успешно!');
        form.reset();
        submitBtn.textContent = originalText;
        submitBtn.disabled = false;
    }, 1500);
}

function smoothScroll(event) {
    event.preventDefault();
    const targetId = this.getAttribute('href');
    if (targetId === '#') return;
    
    const targetElement = document.querySelector(targetId);
    if (targetElement) {
        window.scrollTo({
            top: targetElement.offsetTop - 80,
            behavior: 'smooth'
        });
    }
}

function toggleMoreContent(event) {
    const btn = event.target;
    const container = btn.closest('.content-container');
    const hiddenContent = container.querySelector('.hidden-content');
    
    if (hiddenContent.classList.contains('d-none')) {
        hiddenContent.classList.remove('d-none');
        btn.textContent = 'Скрыть';
    } else {
        hiddenContent.classList.add('d-none');
        btn.textContent = 'Показать больше';
    }
}

function setupScrollAnimations() {
    const observerOptions = {
        threshold: 0.1,
        rootMargin: '0px 0px -50px 0px'
    };
    
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('animate__animated', 'animate__fadeInUp');
            }
        });
    }, observerOptions);

    document.querySelectorAll('.animate-on-scroll').forEach(el => {
        observer.observe(el);
    });
}

function loadAnalyticsData() {
    if (!document.querySelector('#analytics-chart')) return;

    console.log('Загрузка аналитических данных...');
}

window.MIAC = {
    initComponents: initComponents,
    loadAnalyticsData: loadAnalyticsData
};


function validateForm(form) {
    const inputs = form.querySelectorAll('[required]');
    let isValid = true;
    
    inputs.forEach(input => {
        if (!input.value.trim()) {
            input.classList.add('is-invalid');
            isValid = false;
        } else {
            input.classList.remove('is-invalid');
        }
    });
    
    return isValid;
}

function showNotification(message, type = 'success') {
    const alert = document.createElement('div');
    alert.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    alert.style.cssText = `
        top: 20px;
        right: 20px;
        z-index: 9999;
        min-width: 300px;
        box-shadow: 0 5px 15px rgba(0,0,0,0.2);
    `;
    
    alert.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(alert);
    
    setTimeout(() => {
        alert.remove();
    }, 5000);
}

async function fetchData(url, options = {}) {
    try {
        const response = await fetch(url, {
            headers: {
                'Content-Type': 'application/json',
                ...options.headers
            },
            ...options
        });
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        return await response.json();
    } catch (error) {
        console.error('Ошибка при загрузке данных:', error);
        showNotification('Ошибка загрузки данных', 'danger');
        throw error;
    }
}

function saveToLocalStorage(key, data) {
    try {
        localStorage.setItem(key, JSON.stringify(data));
    } catch (error) {
        console.error('Ошибка сохранения в localStorage:', error);
    }
}

function loadFromLocalStorage(key) {
    try {
        const data = localStorage.getItem(key);
        return data ? JSON.parse(data) : null;
    } catch (error) {
        console.error('Ошибка загрузки из localStorage:', error);
        return null;
    }
}

function formatDate(date, format = 'ru-RU') {
    const d = new Date(date);
    return d.toLocaleDateString(format, {
        year: 'numeric',
        month: 'long',
        day: 'numeric'
    });
}

function calculateStatistics(data) {
    if (!Array.isArray(data) || data.length === 0) {
        return null;
    }
    
    const sum = data.reduce((a, b) => a + b, 0);
    const avg = sum / data.length;
    const max = Math.max(...data);
    const min = Math.min(...data);
    
    return { sum, avg, max, min, count: data.length };
}

function initializePage() {

    initComponents();

    const page = document.body.id || window.location.pathname.split('/').pop().replace('.html', '');
    
    switch(page) {
        case 'analytics':
            loadAnalyticsData();
            break;
        case 'services':
            initializeServicesPage();
            break;
        case 'documents':
            initializeDocumentsPage();
            break;
        case 'contacts':
            initializeContactsPage();
            break;
    }

    setupEventListeners();
}

function initializeServicesPage() {

    const savedServices = loadFromLocalStorage('services');
    if (savedServices) {
        updateServicesDisplay(savedServices);
    }
}

function initializeDocumentsPage() {

    const downloadHistory = loadFromLocalStorage('downloadHistory') || [];
    if (downloadHistory.length > 0) {
        console.log('История скачиваний:', downloadHistory);
    }
}

function initializeContactsPage() {

    const form = document.getElementById('feedbackForm');
    if (form) {
        form.addEventListener('input', function() {
            const formData = new FormData(this);
            const data = Object.fromEntries(formData);
            saveToLocalStorage('contactFormDraft', data);
        });

        const draft = loadFromLocalStorage('contactFormDraft');
        if (draft) {
            Object.keys(draft).forEach(key => {
                const input = form.querySelector(`[name="${key}"]`);
                if (input) {
                    input.value = draft[key];
                }
            });
        }
    }
}

document.addEventListener('DOMContentLoaded', initializePage);

window.MIAC = {
    ...window.MIAC,
    validateForm,
    showNotification,
    fetchData,
    saveToLocalStorage,
    loadFromLocalStorage,
    formatDate,
    calculateStatistics,
    initializePage
};